<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'AISV_Admin' ) ) :

class AISV_Admin {

	const LICENSE_TRANSIENT_KEY = 'aisv_license_status';
	const LICENSE_TRANSIENT_TTL = 3600; // 1 hour

	public function __construct() {
		add_action( 'admin_menu', [ $this, 'register_menu' ] );
		add_action( 'wp_dashboard_setup', [ $this, 'register_dashboard_widget' ] );
		add_action( 'admin_init', [ $this, 'maybe_send_test_email' ] );
		add_action( 'admin_notices', [ $this, 'show_test_email_notice' ] );
	}

	/** -------------------------
	 * Admin Menu
	 * --------------------------*/
	public function register_menu() {
		add_menu_page(
			__( 'AI Verification', 'ai-student-verification' ),
			__( 'AI Verification', 'ai-student-verification' ),
			'manage_options',
			'aisv-settings',
			[ $this, 'render_settings_page' ],
			AISV_URL . 'assets/icons/aisc-wp-icon.svg',
			58
		);

		add_submenu_page(
			'aisv-settings',
			__( 'Settings & License', 'ai-student-verification' ),
			__( 'Settings & License', 'ai-student-verification' ),
			'manage_options',
			'aisv-settings',
			[ $this, 'render_settings_page' ]
		);

		add_submenu_page(
			'aisv-settings',
			__( 'Email Templates', 'ai-student-verification' ),
			__( 'Email Templates', 'ai-student-verification' ),
			'manage_options',
			'aisv-emails',
			[ $this, 'render_emails_page' ]
		);

		add_submenu_page(
			'aisv-settings',
			__( 'Verification Logs', 'ai-student-verification' ),
			__( 'Logs', 'ai-student-verification' ),
			'manage_options',
			'aisv-logs',
			[ 'AISV_Logs', 'render_logs_page' ]
		);
	}

	/** -------------------------
	 * Settings Page
	 * --------------------------*/
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$license_key = get_option( 'aisv_license_key' );
		$license     = $this->get_license_status_cached( $license_key ); // cached license lookup
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'AI Student Verification - Settings & License', 'ai-student-verification' ); ?></h1>

			<form method="post" action="options.php">
				<?php settings_fields( 'aisv_settings_group_core' ); ?>

				<!-- License Block -->
				<div class="aisv-setting aisv-license-block">
					<h2><?php esc_html_e( 'License', 'ai-student-verification' ); ?></h2>

					<div style="display:flex;align-items:center;gap:12px;flex-wrap:wrap;">
						<label><strong><?php esc_html_e( 'License Key:', 'ai-student-verification' ); ?></strong></label>
						<input type="text" name="aisv_license_key"
							value="<?php echo esc_attr( $license_key ); ?>"
							class="regular-text" style="min-width:280px;" />

						<?php
						// License status UI (cached)
						if ( $license_key ) {
							$this->render_license_status_ui( $license );
						} else {
							echo '<div class="aisv-no-license" style="margin-top:10px;">';
							echo '<a href="' . esc_url( 'https://bdsplugins.com/ai-student-verification/#aisv-pricing' ) . '" target="_blank" class="aisv-upgrade-link">';
							esc_html_e( 'Get License', 'ai-student-verification' );
							echo '</a></div>';
						}
						?>
					</div>
				</div>

				<!-- Coupons Block -->
				<div class="aisv-setting">
					<h2>
						<?php esc_html_e( 'Coupons', 'ai-student-verification' ); ?>
						<?php
						$coupon_list  = get_option( 'aisv_coupon_list', '' );
						$coupons      = array_filter( array_map( 'trim', explode( "\n", $coupon_list ) ) );
						$coupons_left = count( $coupons );
						?>
						<span style="font-size:12px;font-weight:500;color:#444;">
							<?php
							printf(
								/* translators: %d: number of coupons */
								esc_html__( 'Coupons left: %d', 'ai-student-verification' ),
								intval( $coupons_left )
							);
							?>
						</span>
					</h2>
					<textarea name="aisv_coupon_list" rows="6"><?php echo esc_textarea( $coupon_list ); ?></textarea>
					<p class="description"><?php esc_html_e( 'Enter one coupon per line.', 'ai-student-verification' ); ?></p>
				</div>

				<!-- Verification Settings -->
				<div class="aisv-setting">
					<h2><?php esc_html_e( 'Verification Settings', 'ai-student-verification' ); ?></h2>

					<label><strong><?php esc_html_e( 'Verification Mode', 'ai-student-verification' ); ?></strong></label>
					<select name="aisv_verification_mode">
						<option value="strict" <?php selected( get_option( 'aisv_verification_mode', 'loose' ), 'strict' ); ?>>
							<?php esc_html_e( 'Strict', 'ai-student-verification' ); ?>
						</option>
						<option value="loose" <?php selected( get_option( 'aisv_verification_mode', 'loose' ), 'loose' ); ?>>
							<?php esc_html_e( 'Loose', 'ai-student-verification' ); ?>
						</option>
					</select>

					<span class="aisv-tooltip" style="margin-left:6px;">
						<span class="dashicons dashicons-editor-help" aria-hidden="true"></span>
						<span class="aisv-tooltip-text">
							<strong><?php esc_html_e( 'Loose', 'ai-student-verification' ); ?></strong>: <?php esc_html_e( 'Accepts IDs with minor variations or lower confidence.', 'ai-student-verification' ); ?><br><br>
							<strong><?php esc_html_e( 'Strict', 'ai-student-verification' ); ?></strong>: <?php esc_html_e( 'Requires exact matches and higher confidence from AI verification.', 'ai-student-verification' ); ?>
						</span>
					</span>

					<br><br>

					<label><strong><?php esc_html_e( 'Re-apply Limit', 'ai-student-verification' ); ?></strong></label>
					<input type="number" name="aisv_reapply_limit_days"
						value="<?php echo esc_attr( get_option( 'aisv_reapply_limit_days', 365 ) ); ?>"
						min="0" style="width:100px;" />
					<?php esc_html_e( 'days', 'ai-student-verification' ); ?>

					<span class="aisv-tooltip" style="margin-left:6px;">
						<span class="dashicons dashicons-editor-help" aria-hidden="true"></span>
						<span class="aisv-tooltip-text">
							<?php esc_html_e( 'Minimum number of days before a student can re-apply for verification.', 'ai-student-verification' ); ?>
						</span>
					</span>
				</div>

				<!-- Form Theme Block -->
				<div class="aisv-setting">
					<h2><?php esc_html_e( 'Form Theme', 'ai-student-verification' ); ?></h2>
					<?php
					$selected_theme = get_option( 'aisv_form_theme', 'card' );
					$themes         = [
						'card'    => __( 'Default', 'ai-student-verification' ),
						'minimal' => __( 'Minimal', 'ai-student-verification' ),
						'inline'  => __( 'Inline', 'ai-student-verification' ),
						'glass'   => __( 'Glass', 'ai-student-verification' ),
						'dark'    => __( 'Dark', 'ai-student-verification' ),
					];
					?>
					<select name="aisv_form_theme" id="aisv-form-theme">
						<?php foreach ( $themes as $key => $label ) : ?>
							<option value="<?php echo esc_attr( $key ); ?>" <?php selected( $selected_theme, $key ); ?>>
								<?php echo esc_html( $label ); ?>
							</option>
						<?php endforeach; ?>
					</select>

					<p class="description">
						<?php esc_html_e( 'Choose the default theme for the [aisv_verification_form] shortcode.', 'ai-student-verification' ); ?><br>
						<a href="<?php echo esc_url( 'https://bdsplugins.com/aisv-form-themes' ); ?>" target="_blank">
							<?php esc_html_e( 'See previews of all form themes', 'ai-student-verification' ); ?>
						</a>
					</p>
				</div>

				<!-- Debugging -->
				<div class="aisv-setting">
					<h2><?php esc_html_e( 'Debugging', 'ai-student-verification' ); ?></h2>
					<label>
						<input type="hidden" name="aisv_send_admin_copy" value="0" />
						<input type="checkbox" name="aisv_send_admin_copy" value="1" <?php checked( get_option( 'aisv_send_admin_copy', 0 ), 1 ); ?> />
						<?php esc_html_e( 'Send a copy of all student verification emails to admin', 'ai-student-verification' ); ?>
						(<code><?php echo esc_html( get_option( 'admin_email' ) ); ?></code>)
					</label>
				</div>

				<?php submit_button(); ?>

				<!-- Disclaimer Note -->
				<div class="aisv-disclaimer" style="margin-top:20px; font-size:13px; color:#555; border-left:3px solid #278dd2; padding-left:10px; max-width:700px;">
					<p>
						<strong><?php esc_html_e( 'Privacy Notice:', 'ai-student-verification' ); ?></strong>
						<?php esc_html_e( 'The AISV plugin does not collect or store uploaded IDs. Files are sent directly from your WordPress site to the Google Gemini API (Paid version) for verification and immediately discarded. Site owners remain responsible for their own GDPR compliance and data handling practices.', 'ai-student-verification' ); ?>
						<a href="<?php echo esc_url( 'https://ai.google.dev/gemini-api/terms#data-use-paid' ); ?>" target="_blank" rel="noopener">
							<?php esc_html_e( 'Learn more', 'ai-student-verification' ); ?>
						</a>
					</p>
				</div>
			</form>
		</div>
		<?php
	}

	/** Render the license status UI chunk on settings page */
	private function render_license_status_ui( $data ) {
		// If we have no response, show a gentle message
		if ( empty( $data ) || ( isset( $data['error'] ) && ! isset( $data['valid'] ) ) ) {
			echo '<p style="color:#d63638;">' . esc_html__( 'Could not validate license. Please check your connection or try again later.', 'ai-student-verification' ) . '</p>';
			return;
		}

		// Errors from server
		if ( isset( $data['error'] ) ) {
			$error_msg = sanitize_text_field( $data['error'] );

			if ( 'site_already_has_license' === $error_msg ) {
				$existing_plan = ucfirst( sanitize_text_field( $data['plan'] ?? 'another' ) );
				echo '<p style="color:red; max-width:600px;">' .
					sprintf(
						/* translators: %s: plan name */
						esc_html__( 'This site already has an active %s license. Please upgrade your existing license instead of adding another.', 'ai-student-verification' ),
						'<strong>' . esc_html( $existing_plan ) . '</strong>'
					) . '</p>';
			} elseif ( 'wrong_site' === $error_msg ) {
				echo '<p style="color:red;">' . esc_html__( 'This license is bound to another site. Please use the correct site or reset the license in your account dashboard.', 'ai-student-verification' ) . '</p>';
			} elseif ( 'expired' === $error_msg ) {
				$renew_url = 'https://bdsplugins.com/account';
				echo '<p style="color:red;">' .
					sprintf(
						wp_kses(
							/* translators: %s is a URL */
							__( 'Your license has expired. <a href="%s" target="_blank" rel="noopener">Renew here</a>.', 'ai-student-verification' ),
							[ 'a' => [ 'href' => [], 'target' => [], 'rel' => [] ] ]
						),
						esc_url( $renew_url )
					) . '</p>';
			} else {
				echo '<p style="color:red;">' . esc_html__( 'License invalid:', 'ai-student-verification' ) . ' ' . esc_html( $error_msg ) . '</p>';
			}
			return;
		}

		// Valid license
		if ( isset( $data['valid'] ) && $data['valid'] ) {
			$used = isset( $data['used_verifications'] ) ? (int) $data['used_verifications'] : 0;
			$max  = ( isset( $data['max_verifications'] ) && 'unlimited' !== $data['max_verifications'] )
				? (int) $data['max_verifications']
				: 0;
			$plan = ucfirst( sanitize_text_field( $data['plan_type'] ?? 'Free' ) );

			$pill_class = 'aisv-plan-free';
			if ( 'Pro' === $plan )     { $pill_class = 'aisv-plan-pro'; }
			if ( 'Agency' === $plan )  { $pill_class = 'aisv-plan-agency'; }

			echo '<div class="aisv-plan">';
			echo ' <span class="aisv-plan-pill ' . esc_attr( $pill_class ) . '">' . esc_html( $plan ) . '</span>';

			if ( in_array( $plan, [ 'Free', 'Pro' ], true ) ) {
				echo ' <a href="' . esc_url( 'https://bdsplugins.com/ai-student-verification/#aisv-pricing' ) . '" class="aisv-upgrade-link" target="_blank" rel="noopener">' .
					esc_html__( 'Upgrade', 'ai-student-verification' ) . '</a>';
			}
			echo '</div>';

			// Progress bar
			if ( $max ) {
				$percent   = min( 100, (int) round( ( $used / $max ) * 100 ) );
				$bar_class = $percent >= 90 ? 'aisv-progress-bar danger' : 'aisv-progress-bar';

				echo '<div class="aisv-progress" style="margin-top:12px;">';
				echo '<div class="' . esc_attr( $bar_class ) . '" style="width:' . esc_attr( $percent ) . '%"></div>';
				echo '<span class="aisv-progress-label">' . esc_html( $used . ' / ' . $max ) . '</span>';
				echo '</div>';
			} else {
				echo '<div class="aisv-progress" style="margin-top:12px;">';
				echo '<div class="aisv-progress-bar" style="width:5%"></div>';
				echo '<span class="aisv-progress-label">' . esc_html__( 'Unlimited', 'ai-student-verification' ) . '</span>';
				echo '</div>';
			}
		}
	}

	/** -------------------------
	 * Email Templates Page
	 * --------------------------*/
	public function render_emails_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'AI Student Verification - Email Templates', 'ai-student-verification' ); ?></h1>

			<form method="post" action="options.php">
				<?php settings_fields( 'aisv_settings_group_emails' ); ?>

				<!-- Success Email -->
				<div class="aisv-setting">
					<h2><?php esc_html_e( 'Success Email', 'ai-student-verification' ); ?></h2>
					<label><strong><?php esc_html_e( 'Subject:', 'ai-student-verification' ); ?></strong></label><br>
					<input type="text" name="aisv_email_subject_success"
						value="<?php echo esc_attr( get_option( 'aisv_email_subject_success', 'Your Student Discount Coupon' ) ); ?>"
						class="regular-text" placeholder="<?php esc_attr_e( 'Subject line', 'ai-student-verification' ); ?>" />
					<p class="description"><?php esc_html_e( 'This subject will be sent when verification succeeds.', 'ai-student-verification' ); ?></p>

					<label><strong><?php esc_html_e( 'Content:', 'ai-student-verification' ); ?></strong></label>
					<?php
					wp_editor(
						get_option( 'aisv_email_success_template', '<p>Congratulations! Your coupon is: {coupon}</p>' ),
						'aisv_email_success_template',
						[
							'textarea_name' => 'aisv_email_success_template',
							'textarea_rows' => 8,
							'teeny'         => true,
							'media_buttons' => false,
						]
					);

					$success_url = wp_nonce_url(
						add_query_arg( [ 'aisv_test_email' => 'success' ] ),
						'aisv_send_test_email',
						'aisv_nonce'
					);
					?>
					<p>
						<a href="<?php echo esc_url( $success_url ); ?>"
							class="button button-secondary"><?php esc_html_e( 'Send Test Success Email', 'ai-student-verification' ); ?></a>
					</p>
				</div>

				<!-- Failure Email -->
				<div class="aisv-setting">
					<h2><?php esc_html_e( 'Failure Email', 'ai-student-verification' ); ?></h2>
					<label><strong><?php esc_html_e( 'Subject:', 'ai-student-verification' ); ?></strong></label><br>
					<input type="text" name="aisv_email_subject_failure"
						value="<?php echo esc_attr( get_option( 'aisv_email_subject_failure', 'Student Verification Failed' ) ); ?>"
						class="regular-text" placeholder="<?php esc_attr_e( 'Subject line', 'ai-student-verification' ); ?>" />
					<p class="description"><?php esc_html_e( 'This subject will be sent when verification fails.', 'ai-student-verification' ); ?></p>

					<label><strong><?php esc_html_e( 'Content:', 'ai-student-verification' ); ?></strong></label>
					<?php
					wp_editor(
						get_option( 'aisv_email_failure_template', '<p>Sorry, we could not verify your student ID for {email}.</p>' ),
						'aisv_email_failure_template',
						[
							'textarea_name' => 'aisv_email_failure_template',
							'textarea_rows' => 8,
							'teeny'         => true,
							'media_buttons' => false,
						]
					);

					$failure_url = wp_nonce_url(
						add_query_arg( [ 'aisv_test_email' => 'failure' ] ),
						'aisv_send_test_email',
						'aisv_nonce'
					);
					?>
					<p>
						<a href="<?php echo esc_url( $failure_url ); ?>"
							class="button button-secondary"><?php esc_html_e( 'Send Test Failure Email', 'ai-student-verification' ); ?></a>
					</p>
				</div>

				<!-- Duplicate Email -->
				<div class="aisv-setting">
					<h2><?php esc_html_e( 'Already Verified Email', 'ai-student-verification' ); ?></h2>
					<label><strong><?php esc_html_e( 'Subject:', 'ai-student-verification' ); ?></strong></label><br>
					<input type="text" name="aisv_email_subject_duplicate"
						value="<?php echo esc_attr( get_option( 'aisv_email_subject_duplicate', 'Student Verification Already Completed' ) ); ?>"
						class="regular-text" placeholder="<?php esc_attr_e( 'Subject line', 'ai-student-verification' ); ?>" />
					<p class="description"><?php esc_html_e( 'This subject will be sent if the student is already verified.', 'ai-student-verification' ); ?></p>

					<label><strong><?php esc_html_e( 'Content:', 'ai-student-verification' ); ?></strong></label>
					<?php
					wp_editor(
						get_option( 'aisv_email_duplicate_template', '<p>Hi {username},</p><p>You were already verified on {last_verified_date}. Your coupon was: {last_coupon}.</p><p>You can re-apply after {reapply_limit_days} days.</p>' ),
						'aisv_email_duplicate_template',
						[
							'textarea_name' => 'aisv_email_duplicate_template',
							'textarea_rows' => 8,
							'teeny'         => true,
							'media_buttons' => false,
						]
					);

					$duplicate_url = wp_nonce_url(
						add_query_arg( [ 'aisv_test_email' => 'duplicate' ] ),
						'aisv_send_test_email',
						'aisv_nonce'
					);
					?>
					<p>
						<a href="<?php echo esc_url( $duplicate_url ); ?>"
							class="button button-secondary"><?php esc_html_e( 'Send Test Duplicate Email', 'ai-student-verification' ); ?></a>
					</p>
				</div>

				<div class="aisv-setting">
					<h2><?php esc_html_e( 'Available Tags', 'ai-student-verification' ); ?></h2>
					<p class="description">
						<?php esc_html_e( 'You can use the following tags in subjects and content:', 'ai-student-verification' ); ?><br>
						<code>{username}</code>, <code>{email}</code>, <code>{coupon}</code>, <code>{site_name}</code>,
						<code>{reapply_limit_days}</code>, <code>{last_verified_date}</code>, <code>{last_coupon}</code>
					</p>
				</div>

				<?php submit_button(); ?>
			</form>
		</div>
		<?php
	}

	/** -------------------------
	 * Test Email Handling with Nonce
	 * --------------------------*/
	public function maybe_send_test_email() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		if ( empty( $_GET['aisv_test_email'] ) ) {
			return;
		}
		if ( empty( $_GET['aisv_nonce'] ) || ! wp_verify_nonce( $_GET['aisv_nonce'], 'aisv_send_test_email' ) ) {
			return;
		}

		$type = sanitize_text_field( wp_unslash( $_GET['aisv_test_email'] ) );
		$to   = get_option( 'admin_email' ); // send test email to admin
		$site = get_bloginfo( 'name' );

		$placeholders = [
			'{username}'           => 'TestUser',
			'{email}'              => 'test@example.com',
			'{coupon}'             => 'TESTCOUPON123',
			'{site_name}'          => $site,
			'{reapply_limit_days}' => 30,
			'{last_verified_date}' => date( 'Y-m-d' ),
			'{last_coupon}'        => 'OLDCODE456',
		];

		if ( 'success' === $type ) {
			$subject = get_option( 'aisv_email_subject_success', 'Your Student Discount Coupon' );
			$body    = get_option( 'aisv_email_success_template', '<p>Congratulations! Your coupon is: {coupon}</p>' );
		} elseif ( 'failure' === $type ) {
			$subject = get_option( 'aisv_email_subject_failure', 'Student Verification Failed' );
			$body    = get_option( 'aisv_email_failure_template', '<p>Sorry, we could not verify your student ID for {email}.</p>' );
		} else {
			$subject = get_option( 'aisv_email_subject_duplicate', 'Student Verification Already Completed' );
			$body    = get_option( 'aisv_email_duplicate_template', '<p>You were already verified…</p>' );
		}

		$subject = str_replace( array_keys( $placeholders ), array_values( $placeholders ), $subject );
		$body    = str_replace( array_keys( $placeholders ), array_values( $placeholders ), $body );

		$sent = wp_mail( $to, $subject, $body, [ 'Content-Type: text/html; charset=UTF-8' ] );

		if ( $sent ) {
			set_transient(
				'aisv_test_email_notice',
				sprintf(
					/* translators: %s: email address */
					__( 'Test email sent to %s', 'ai-student-verification' ),
					$to
				),
				30
			);
		} else {
			set_transient(
				'aisv_test_email_notice',
				sprintf(
					/* translators: %s: email address */
					__( 'Failed to send test email to %s', 'ai-student-verification' ),
					$to
				),
				30
			);
		}

		// Redirect to clean URL (remove query param)
		wp_safe_redirect( remove_query_arg( [ 'aisv_test_email', 'aisv_nonce' ] ) );
		exit;
	}

	public function show_test_email_notice() {
		$msg = get_transient( 'aisv_test_email_notice' );
		if ( $msg ) {
			echo '<div class="notice notice-success"><p>' . esc_html( $msg ) . '</p></div>';
			delete_transient( 'aisv_test_email_notice' );
		}
	}

	/** -------------------------
	 * Dashboard Widget
	 * --------------------------*/
	public function register_dashboard_widget() {
		wp_add_dashboard_widget(
			'aisv_dashboard_widget',
			__( 'AI Student Verification Usage', 'ai-student-verification' ),
			[ $this, 'render_dashboard_widget' ]
		);
	}

	public function render_dashboard_widget() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$license_key = get_option( 'aisv_license_key' );
		if ( empty( $license_key ) ) {
			echo '<p>' . esc_html__( 'No license key set in settings.', 'ai-student-verification' ) . '</p>';
			return;
		}

		$data = $this->get_license_status_cached( $license_key );

		if ( empty( $data ) || ! isset( $data['valid'] ) ) {
			echo '<p>' . esc_html__( 'Could not validate license. Please check your connection or license key.', 'ai-student-verification' ) . '</p>';
			return;
		}

		if ( ! $data['valid'] ) {
			echo '<p>' . esc_html__( 'License invalid:', 'ai-student-verification' ) . ' ' . esc_html( $data['error'] ?? '' ) . '</p>';
			return;
		}

		// Usage info
		$used = (int) ( $data['used_verifications'] ?? 0 );
		$max  = ( isset( $data['max_verifications'] ) && 'unlimited' !== $data['max_verifications'] )
			? (int) $data['max_verifications']
			: 0;
		$plan = ucfirst( sanitize_text_field( $data['plan_type'] ?? 'Free' ) );

		$pill_class = 'aisv-plan-free';
		if ( 'Pro' === $plan )     { $pill_class = 'aisv-plan-pro'; }
		if ( 'Agency' === $plan )  { $pill_class = 'aisv-plan-agency'; }

		// Coupons left
		$coupon_list  = get_option( 'aisv_coupon_list', '' );
		$coupons      = array_filter( array_map( 'trim', explode( "\n", $coupon_list ) ) );
		$coupons_left = count( $coupons );

		// Header row (plan left, coupons right)
		echo '<div class="aisv-header-row">';
		echo '<div class="aisv-plan">';
		echo '<strong>' . esc_html__( 'Plan:', 'ai-student-verification' ) . '</strong> ';
		echo '<span class="aisv-plan-pill ' . esc_attr( $pill_class ) . '">' . esc_html( $plan ) . '</span>';

		if ( in_array( $plan, [ 'Free', 'Pro' ], true ) ) {
			echo ' <a href="' . esc_url( 'https://bdsplugins.com/ai-student-verification/#aisv-pricing' ) . '" class="aisv-upgrade-link" target="_blank" rel="noopener">' .
				esc_html__( 'Upgrade', 'ai-student-verification' ) . '</a>';
		}
		echo '</div>';

		echo '<div class="aisv-coupons">' .
			sprintf(
				/* translators: %d: number of coupons left */
				esc_html__( 'Coupons left: %d', 'ai-student-verification' ),
				intval( $coupons_left )
			)
			. '</div>';
		echo '</div>';

		// Progress bar
		if ( $max ) {
			$percent   = min( 100, (int) round( ( $used / $max ) * 100 ) );
			$bar_class = $percent >= 90 ? 'aisv-progress-bar danger' : 'aisv-progress-bar';

			echo '<div class="aisv-progress">';
			echo '<div class="' . esc_attr( $bar_class ) . '" style="width:' . esc_attr( $percent ) . '%"></div>';
			echo '<span class="aisv-progress-label">' . esc_html( $used . ' / ' . $max ) . '</span>';
			echo '</div>';
		} else {
			echo '<div class="aisv-progress">';
			echo '<div class="aisv-progress-bar" style="width:5%"></div>';
			echo '<span class="aisv-progress-label">' . esc_html__( 'Unlimited', 'ai-student-verification' ) . '</span>';
			echo '</div>';
		}

		// Recent Verifications Preview
		global $wpdb;
		$table = $wpdb->prefix . 'aisv_logs';
		// Safe here (no user input). If table name changed dynamically, use $wpdb->prepare.
		$logs  = $wpdb->get_results( "SELECT email, status, coupon, created_at FROM {$table} ORDER BY id DESC LIMIT 5" );

		echo '<h4 style="margin-top:15px;">' . esc_html__( 'Recent Verifications', 'ai-student-verification' ) . '</h4>';
		if ( empty( $logs ) ) {
			echo '<p>' . esc_html__( 'No verifications yet.', 'ai-student-verification' ) . '</p>';
		} else {
			echo '<table class="widefat striped" style="font-size:12px;margin-top:8px;">';
			echo '<thead><tr><th>' . esc_html__( 'Email', 'ai-student-verification' ) . '</th><th>' . esc_html__( 'Status', 'ai-student-verification' ) . '</th><th>' . esc_html__( 'Coupon', 'ai-student-verification' ) . '</th><th>' . esc_html__( 'Date', 'ai-student-verification' ) . '</th></tr></thead><tbody>';
				foreach ( $logs as $log ) {
					$status_color = ( 'approved' === $log->status ) ? 'green' : ( ( 'declined' === $log->status ) ? 'red' : 'orange' );
					$email_display = $log->email;
					if ( strlen( $email_display ) > 18 ) {
						$email_display = substr( $email_display, 0, 15 ) . '...';
					}
					$date_display = date_i18n( 'M j, H:i', strtotime( $log->created_at ) );
					echo '<tr>';
					echo '<td title="' . esc_attr( $log->email ) . '">' . esc_html( $email_display ) . '</td>';
					echo '<td style="color:' . esc_attr( $status_color ) . '">' . esc_html( ucfirst( $log->status ) ) . '</td>';
					echo '<td>' . esc_html( $log->coupon ) . '</td>';
					echo '<td>' . esc_html( $date_display ) . '</td>';
					echo '</tr>';
				}
			echo '</tbody></table>';
		}

		echo '<p><a href="' . esc_url( admin_url( 'admin.php?page=aisv-logs' ) ) . '" class="button">' . esc_html__( 'View all', 'ai-student-verification' ) . '</a></p>';
	}

	/** -------------------------
	 * License Status Helper (Cached)
	 * --------------------------*/
	private function get_license_status_cached( $license_key ) {
		$license_key = (string) $license_key;

		if ( empty( $license_key ) ) {
			return null;
		}

		// Include key+site in cache key to avoid stale cross-site cache
		$cache_key = self::LICENSE_TRANSIENT_KEY . '_' . md5( $license_key . '|' . home_url() );
		$data      = get_transient( $cache_key );

		if ( false !== $data ) {
			return $data;
		}

		$response = wp_remote_get(
			add_query_arg(
				[
					'site_url' => home_url(),
				],
				'https://bdsplugins.com/api/check-license.php'
			),
			[
				'headers' => [
					'Authorization' => 'Bearer ' . $license_key,
					'X-Site-Url'    => home_url(),
				],
				'timeout' => 15,
			]
		);

		if ( is_wp_error( $response ) ) {
			// Cache a minimal error for a short period to avoid hammering
			set_transient( $cache_key, [ 'error' => 'connection_error' ], 300 );
			return [ 'error' => 'connection_error' ];
		}

		$decoded = json_decode( wp_remote_retrieve_body( $response ), true );

		// Cache decoded (valid or invalid) to reduce server load
		if ( is_array( $decoded ) ) {
			set_transient( $cache_key, $decoded, self::LICENSE_TRANSIENT_TTL );
			return $decoded;
		}

		set_transient( $cache_key, [ 'error' => 'invalid_json' ], 300 );
		return [ 'error' => 'invalid_json' ];
	}
}

endif;

new AISV_Admin();
