<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AISV_Hooks {

	private $mail_headers = [ 'Content-Type: text/html; charset=UTF-8' ];

	public function __construct() {
		add_action( 'wpforms_process_complete', [ $this, 'handle_verification' ], 10, 4 );
	}

	public function handle_verification( $fields, $entry, $form_data, $entry_id ) {
		$current_id = isset( $form_data['id'] ) ? (int) $form_data['id'] : 0;
		$enabled    = ! empty( $form_data['settings']['aisv_enabled'] );
		if ( ! $enabled && ( ! class_exists( 'AISV_WPForms' ) || ! AISV_WPForms::is_enabled_for_form( $current_id ) ) ) {
			return;
		}

		$student_email = '';
		$file_path     = '';

		foreach ( $fields as $field ) {
			if ( isset( $field['type'] ) && 'email' === $field['type'] && empty( $student_email ) ) {
				$student_email = sanitize_email( $field['value'] );
			}
			if ( isset( $field['type'] ) && 'file-upload' === $field['type'] && ! empty( $field['value'] ) ) {
				$uploads = maybe_unserialize( $field['value'] );
				if ( is_array( $uploads ) && ! empty( $uploads[0] ) ) {
					$file_path = $uploads[0];
				} elseif ( is_string( $field['value'] ) ) {
					$file_path = $field['value'];
				}
			}
		}

		if ( ! $student_email || ! $file_path ) {
			return;
		}

		self::process_verification( $student_email, $file_path );
	}

	public static function process_verification( $student_email, $file_path ) {
		$instance = new self();
		return $instance->run_verification( $student_email, $file_path );
	}

	private function run_verification( $student_email, $file_path ) {
		global $wpdb;
		$table = $wpdb->prefix . 'aisv_logs';

		$reapply_limit_days = (int) get_option( 'aisv_reapply_limit_days', 365 );
		if ( $reapply_limit_days > 0 ) {
			$recent_approval = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT coupon, created_at FROM {$table}
					WHERE email = %s AND status = 'approved'
					ORDER BY created_at DESC LIMIT 1",
					$student_email
				)
			);

			if ( $recent_approval ) {
				$last_time = strtotime( $recent_approval->created_at );
				$diff_days = floor( ( time() - $last_time ) / DAY_IN_SECONDS );
				if ( $diff_days < $reapply_limit_days ) {
					$subject  = sanitize_text_field( get_option( 'aisv_email_subject_duplicate', __( 'Student Verification Already Completed', 'ai-student-verification' ) ) );
					$template = get_option( 'aisv_email_duplicate_template', '<p>' . __( 'You already verified on {last_verified_date}. Your coupon: {last_coupon}', 'ai-student-verification' ) . '</p>' );
					$user     = get_user_by( 'email', $student_email );
					$username = $user ? ( $user->display_name ?: $user->user_login ) : strstr( $student_email, '@', true );

					$replacements = [
						'{username}'           => $username,
						'{last_verified_date}' => date_i18n( get_option( 'date_format' ), $last_time ),
						'{last_coupon}'        => $recent_approval->coupon,
						'{reapply_limit_days}' => $reapply_limit_days,
						'{site_name}'          => get_bloginfo( 'name' ),
					];
					$body = str_replace( array_keys( $replacements ), array_values( $replacements ), $template );

					wp_mail( $student_email, $subject, wpautop( $body ), $this->mail_headers );
					if ( get_option( 'aisv_send_admin_copy', 0 ) ) {
						wp_mail( get_option( 'admin_email' ), $subject, wpautop( $body ), $this->mail_headers );
					}

					$wpdb->insert(
						$table,
						[
							'email'    => $student_email,
							'coupon'   => $recent_approval->coupon,
							'status'   => 'duplicate',
							'ai_reply' => 'Re-apply blocked (' . $diff_days . ' days)',
						]
					);
					return [ 'status' => 'duplicate', 'coupon' => $recent_approval->coupon, 'message' => __( 'Already verified', 'ai-student-verification' ) ];
				}
			}
		}

		// --- API verification ---
		$api_result = $this->verify_student_id( $file_path );
		$verified   = ! empty( $api_result['verified'] );
		$ai_reply   = isset( $api_result['reply'] ) ? sanitize_text_field( $api_result['reply'] ) : '';

		if ( ! empty( $api_result['error'] ) ) {
			$wpdb->insert(
				$table,
				[
					'email'    => $student_email,
					'coupon'   => '',
					'status'   => 'error',
					'ai_reply' => $ai_reply ?: sanitize_text_field( $api_result['error'] ),
				]
			);

			$error_code = $api_result['error'];

			// Friendly messages for common cases
			switch ( $error_code ) {
				case 'file_type_not_allowed':
					$message = __( 'Unsupported file type. Please upload a JPG, PNG, or PDF file.', 'ai-student-verification' );
					break;
				case 'file_too_large':
					$message = __( 'File too large. Please upload a file under 3MB.', 'ai-student-verification' );
					break;
				case 'file_unreadable':
					$message = __( 'Could not read the uploaded file. Please try again.', 'ai-student-verification' );
					break;
				default:
					$message = __( 'Verification failed. Please make sure your ID is clear and in a supported format (JPG, PNG, or PDF).', 'ai-student-verification' );
					break;
			}

			return [ 'status' => 'error', 'coupon' => null, 'message' => $message ];
		}

		// --- Email templates ---
		$subject_success = sanitize_text_field( get_option( 'aisv_email_subject_success', __( 'Your Student Discount Coupon', 'ai-student-verification' ) ) );
		$subject_failure = sanitize_text_field( get_option( 'aisv_email_subject_failure', __( 'Student Verification Failed', 'ai-student-verification' ) ) );
		$success_tpl     = get_option( 'aisv_email_success_template', '<p>' . __( 'Congrats! Your coupon: {coupon}', 'ai-student-verification' ) . '</p>' );
		$failure_tpl     = get_option( 'aisv_email_failure_template', '<p>' . __( 'Sorry, we could not verify your ID for {email}.', 'ai-student-verification' ) . '</p>' );

		$user     = get_user_by( 'email', $student_email );
		$username = $user ? ( $user->display_name ?: $user->user_login ) : strstr( $student_email, '@', true );

		$replacements = [
			'{coupon}'    => '',
			'{email}'     => $student_email,
			'{username}'  => $username,
			'{site_name}' => get_bloginfo( 'name' ),
		];

		// --- Success case ---
		if ( $verified ) {
			$coupon = $this->get_next_coupon();
			if ( $coupon ) {
				$replacements['{coupon}'] = $coupon;
				$body                     = str_replace( array_keys( $replacements ), array_values( $replacements ), $success_tpl );
				wp_mail( $student_email, $subject_success, wpautop( $body ), $this->mail_headers );
				if ( get_option( 'aisv_send_admin_copy', 0 ) ) {
					wp_mail( get_option( 'admin_email' ), $subject_success, wpautop( $body ), $this->mail_headers );
				}
				$wpdb->insert(
					$table,
					[
						'email'    => $student_email,
						'coupon'   => $coupon,
						'status'   => 'approved',
						'ai_reply' => $ai_reply,
					]
				);
				return [ 'status' => 'approved', 'coupon' => $coupon, 'message' => __( 'Verification successful', 'ai-student-verification' ) ];
			}
			$wpdb->insert(
				$table,
				[
					'email'    => $student_email,
					'coupon'   => '',
					'status'   => 'error',
					'ai_reply' => 'no_coupons_left',
				]
			);
			return [ 'status' => 'error', 'coupon' => null, 'message' => __( 'No coupons left', 'ai-student-verification' ) ];
		}

		// --- Failure case ---
		$body = str_replace( array_keys( $replacements ), array_values( $replacements ), $failure_tpl );
		wp_mail( $student_email, $subject_failure, wpautop( $body ), $this->mail_headers );
		if ( get_option( 'aisv_send_admin_copy', 0 ) ) {
			wp_mail( get_option( 'admin_email' ), $subject_failure, wpautop( $body ), $this->mail_headers );
		}

		$wpdb->insert(
			$table,
			[
				'email'    => $student_email,
				'coupon'   => '',
				'status'   => 'declined',
				'ai_reply' => $ai_reply,
			]
		);

		return [ 'status' => 'declined', 'coupon' => null, 'message' => __( 'Verification declined', 'ai-student-verification' ) ];
	}

	// --- Helpers ---
	private function verify_student_id( $file_path ) {
		$endpoint = 'https://bdsplugins.com/api/verify-student.php';
		$license  = get_option( 'aisv_license_key', '' );
		$mode     = get_option( 'aisv_verification_mode', 'loose' );

		// ✅ NEW: handle array input from shortcode uploads
		if ( is_array( $file_path ) && ! empty( $file_path['tmp_name'] ) ) {
			$uploads = wp_get_upload_dir();
			$tempdir = trailingslashit( $uploads['basedir'] ) . 'aisv-temp/';
			if ( ! file_exists( $tempdir ) ) {
				wp_mkdir_p( $tempdir );
			}
			$target = $tempdir . wp_unique_filename( $tempdir, $file_path['name'] );
			if ( move_uploaded_file( $file_path['tmp_name'], $target ) ) {
				$file_path = $target;
			}
		}

		$local = $this->aisv_resolve_local_path( $file_path );
		if ( ! $local || ! is_readable( $local ) ) {
			update_option( 'aisv_last_error', 'file_unreadable' );
			return [ 'verified' => false, 'reply' => __( 'File not readable', 'ai-student-verification' ), 'error' => 'file_unreadable' ];
		}

		$max_size = 3 * 1024 * 1024;
		if ( filesize( $local ) > $max_size ) {
			update_option( 'aisv_last_error', 'file_too_large' );
			return [ 'verified' => false, 'reply' => __( 'File too large', 'ai-student-verification' ), 'error' => 'file_too_large' ];
		}

		$allowed_types = [ 'image/jpeg', 'image/png', 'application/pdf' ];
		$mime = wp_check_filetype( basename( $local ) );
		if ( empty( $mime['type'] ) || ! in_array( $mime['type'], $allowed_types, true ) ) {
			update_option( 'aisv_last_error', 'file_type_not_allowed' );
			return [ 'verified' => false, 'reply' => __( 'Unsupported file type', 'ai-student-verification' ), 'error' => 'file_type_not_allowed' ];
		}

        $contents = @file_get_contents( $local );
        if ( false === $contents ) {
            update_option( 'aisv_last_error', 'file_unreadable' );
            return [ 'verified' => false, 'reply' => __( 'File not readable', 'ai-student-verification' ), 'error' => 'file_unreadable' ];
        }
        if ( file_exists( $local ) ) {
            @unlink( $local );
        }

		$payload = [
			'file' => base64_encode( $contents ),
			'mode' => $mode,
		];

		$response = wp_remote_post(
			$endpoint,
			[
				'headers' => [
					'Authorization' => 'Bearer ' . $license,
					'Content-Type'  => 'application/json',
				],
				'body'    => wp_json_encode( $payload ),
				'timeout' => 30,
			]
		);

		if ( is_wp_error( $response ) ) {
			$msg = $response->get_error_message();
			update_option( 'aisv_last_error', 'API error: ' . $msg );
			return [ 'verified' => false, 'reply' => __( 'API error', 'ai-student-verification' ), 'error' => $msg ];
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );
		if ( ! is_array( $body ) ) {
			update_option( 'aisv_last_error', 'bad_json' );
			return [ 'verified' => false, 'reply' => __( 'Bad API response', 'ai-student-verification' ), 'error' => 'bad_json' ];
		}

		if ( ! empty( $body['error'] ) ) {
			update_option( 'aisv_last_error', sanitize_text_field( $body['error'] ) );
		}

		return $body;
	}

	private function aisv_resolve_local_path( $file_path ) {
		if ( empty( $file_path ) ) {
			return '';
		}

		$file_path = wp_unslash( $file_path );
		$uploads   = wp_get_upload_dir();
		$norm      = wp_normalize_path( $file_path );

		if ( file_exists( $norm ) && is_readable( $norm ) ) {
			return $norm;
		}

		if ( filter_var( $file_path, FILTER_VALIDATE_URL ) ) {
			if ( strpos( $file_path, $uploads['baseurl'] ) === 0 ) {
				$rel       = substr( $file_path, strlen( $uploads['baseurl'] ) );
				$candidate = wp_normalize_path( $uploads['basedir'] . $rel );
				if ( file_exists( $candidate ) && is_readable( $candidate ) ) {
					return $candidate;
				}
			}
			if ( function_exists( 'download_url' ) ) {
				$tmp = download_url( $file_path, 30 );
				if ( ! is_wp_error( $tmp ) && is_readable( $tmp ) ) {
					return $tmp;
				}
			}
			return '';
		}

		$candidate = wp_normalize_path( trailingslashit( $uploads['basedir'] ) . ltrim( $file_path, '/\\' ) );
		if ( file_exists( $candidate ) && is_readable( $candidate ) ) {
			return $candidate;
		}

		return '';
	}

	private function get_next_coupon() {
		$lock_key = 'aisv_coupon_lock';
		if ( get_transient( $lock_key ) ) return false;
		set_transient( $lock_key, 1, 5 );

		$list    = (string) get_option( 'aisv_coupon_list', '' );
		$coupons = array_values( array_filter( array_map( 'trim', preg_split( "/\r\n|\n|\r/", $list ) ) ) );
		if ( empty( $coupons ) ) { delete_transient( $lock_key ); return false; }

		$coupon = array_shift( $coupons );
		update_option( 'aisv_coupon_list', implode( "\n", $coupons ), false );

		delete_transient( $lock_key );
		return $coupon;
	}
}

new AISV_Hooks();
