<?php
if ( ! defined('ABSPATH') ) exit;

class AISV_WPForms_Admin {

    public function __construct() {
        // Add toggle setting to WPForms builder
        add_action( 'wpforms_form_settings_panel_content', [ $this, 'add_toggle_setting' ], 10, 1 );
        // Save toggle setting with form
        add_filter( 'wpforms_builder_save_form_args', [ $this, 'save_toggle_setting' ], 10, 1 );
        // Add default setting for new forms
        add_filter( 'wpforms_form_settings_defaults', [ $this, 'add_setting_default' ], 10, 1 );
    }

    /**
     * Add the AISV toggle to WPForms Settings panel.
     *
     * @param WPForms_Builder_Panel_Settings $panel Builder panel object.
     */
    public function add_toggle_setting( $panel ) {
        list( $enabled ) = $this->resolve_enabled_state( $panel );
        ?>
        <div class="wpforms-panel-field wpforms-panel-field-toggle">
            <span class="wpforms-toggle-control">
                <input type="checkbox"
                       id="wpforms-panel-field-settings-aisv_enabled"
                       name="settings[aisv_enabled]"
                       value="1" <?php checked( $enabled, 1 ); ?>>
                <label class="wpforms-toggle-control-icon"
                       for="wpforms-panel-field-settings-aisv_enabled"></label>
                <label class="wpforms-toggle-control-label"
                       for="wpforms-panel-field-settings-aisv_enabled">
                    <?php esc_html_e( 'Enable AI Student Verification for this form', 'ai-student-verification' ); ?>
                </label>
            </span>
        </div>
        <?php
    }

    /**
     * Save our custom setting when a form is saved.
     *
     * @param array $form Full form data.
     * @return array
     */
    public function save_toggle_setting( $form ) {
        $form['settings']['aisv_enabled'] = ! empty( $_POST['settings']['aisv_enabled'] ) ? absint( $_POST['settings']['aisv_enabled'] ) : 0;
        return $form;
    }

    /**
     * Provide a default value for new forms.
     *
     * @param array $settings Form settings.
     * @return array
     */
    public function add_setting_default( $settings ) {
        if ( ! isset( $settings['aisv_enabled'] ) ) {
            $settings['aisv_enabled'] = 0;
        }
        return $settings;
    }

    /**
     * Determine toggle state from builder panel or stored DB value.
     *
     * @param mixed $panel Builder panel object or null.
     * @return array [ enabled(int 0/1) ]
     */
    private function resolve_enabled_state( $panel ) {
        // Prefer builder panel object
        if (
            is_object( $panel )
            && isset( $panel->form )
            && is_object( $panel->form )
            && isset( $panel->form->settings )
            && is_array( $panel->form->settings )
        ) {
            $enabled = ! empty( $panel->form->settings['aisv_enabled'] ) ? 1 : 0;
            return [ $enabled ];
        }

        // Fallback: check stored form settings
        $form_id = 0;
        foreach ( [ 'form_id', 'id', 'view_id' ] as $key ) {
            if ( isset( $_GET[ $key ] ) ) {
                $form_id = absint( $_GET[ $key ] );
                if ( $form_id ) break;
            }
        }

        if ( $form_id ) {
            $post = get_post( $form_id );
            if ( $post && ! empty( $post->post_content ) ) {
                $data = json_decode( $post->post_content, true );
                if ( is_array( $data ) && ! empty( $data['settings'] ) && is_array( $data['settings'] ) ) {
                    $enabled = ! empty( $data['settings']['aisv_enabled'] ) ? 1 : 0;
                    return [ $enabled ];
                }
            }
        }

        return [ 0 ];
    }
}
new AISV_WPForms_Admin();
